/**
* \file: configuration.c
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: authorization level daemon
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#include <sys/mount.h>
#include <string.h>
#include <argp.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <unistd.h>
#include <grp.h>

#include "control/ald.h"
#include "control/configuration.h"
#include "control/configuration_defaults.h"
#include "util/version_info.h"
#include "util/helper.h"
#include "util/cfgfile_parser.h"

static const char *proc_name = NULL;
static bool console_enabled = DEFAULT_CONSOLE_ENABLED;

static const char * const configuration_doc =
		"ADIT authorization level daemon - A daemon for automatically switching security levels in the system.";

/*
 * The options ALD understands.
 */
static const struct argp_option const configuration_options[] =
{
		{"loglevel",  'l', "<loglevel>" ,      0,  "Start the daemon with a given log level. Supported values are: none, error, info, debug",0},
		{"quiet",  'q', NULL ,      0,  "Don't plot any status messages or log message to the console.",0},
		{"version",  'v', NULL ,      0,  "Prints out version information.",0},
		{"config", 'c', "<conf file>", 0, "Load the configuration from given file. [default: /etc/ald.conf]",0},
		{NULL,0,NULL,0,NULL,0}
};

static cfg_item_t loglevel_cfg 			= {.spec = &log_level_cfg_spec };
static cfg_item_t state_cfg 			= {.spec = &state_path_cfg_spec };
static cfg_item_t script_root_cfg 		= {.spec = &script_root_cfg_spec };
static cfg_item_t sigdb_cfg 			= {.spec = &sigdb_cfg_spec };
static cfg_item_t sigdb_key_cfg 		= {.spec = &sigdb_key_cfg_spec };
static cfg_item_t challenge_timeout_cfg 	= {.spec = &challenge_timeout_cfg_spec };
static cfg_item_t replay_trigger_cfg 		= {.spec = &replay_trigger_cfg_spec };
static cfg_item_t level_chng_done_cfg 		= {.spec = &level_change_complete_cfg_spec };
static cfg_item_t script_timeout_cfg 		= {.spec = &script_timeout_cfg_spec };
static cfg_item_t challenge_verify_cfg 		= {.spec = &challenge_verify_key_cfg_spec};
static cfg_item_t conffile_cfg 			= {.spec = &conf_file_cfg_spec };

static cfg_item_t *const ald_cfgs[] = {
		&loglevel_cfg,
		&state_cfg,
		&script_root_cfg,
		&sigdb_cfg,
		&sigdb_key_cfg,
		&challenge_timeout_cfg,
		&replay_trigger_cfg,
		&level_chng_done_cfg,
		&script_timeout_cfg,
		&challenge_verify_cfg,
		&conffile_cfg
};


static error_code_t configuration_parse_args(int argc, char *argv[]);

static char *configuration_filter_help(int key, const char *text, void *input);

static error_t configuration_parse_opt (int key, char *arg, struct argp_state *state);

static void configuration_print_version(void);

//--------------------------------------- API members ------------------------------------
error_code_t configuration_init(int argc, char *argv[])
{
	helper_items_init(ald_cfgs, ARRAY_SIZE(ald_cfgs));

	proc_name = argv[0];

	return configuration_parse_args(argc,argv);
}

void configuration_deinit(void)
{
	helper_items_free(ald_cfgs, ARRAY_SIZE(ald_cfgs));
}

error_code_t configuration_parse_ald_configuration_file(void)
{
	return cfgfile_parser_parse_config_file(helper_get_str_value(&conffile_cfg), ald_cfgs, ARRAY_SIZE(ald_cfgs));
}

const char *configuration_get_proc_name(void)
{
	return proc_name;
}

bool configuration_is_console_enabled(void)
{
	return console_enabled;
}

logger_loglevel_t configuration_get_loglevel(void)
{
	return helper_get_loglevel(&loglevel_cfg);
}

const char *configuration_get_persisted_state_path(void)
{
	return helper_get_str_value(&state_cfg);
}

const char *configuration_get_script_root_dir(void)
{
	return helper_get_str_value(&script_root_cfg);
}

const char *configuration_get_signature_db_path(void)
{
	return helper_get_str_value(&sigdb_cfg);
}
const char *configuration_get_signature_db_pub_key_path(void)
{
	return helper_get_str_value(&sigdb_key_cfg);
}

uint32_t configuration_get_challenge_timeout_ms(void)
{
	return helper_get_U32_value(&challenge_timeout_cfg);
}

uint32_t configuration_get_script_exec_timeout_ms(void)
{
	return helper_get_U32_value(&script_timeout_cfg);
}

const char *configuration_get_replay_level_trigger_path(void)
{
	return helper_get_str_value(&replay_trigger_cfg);
}

const char *configuration_get_level_change_complete_file_path(void)
{
	return helper_get_str_value(&level_chng_done_cfg);
}

uint32_t configuration_get_challenge_verify_key_behaviour(void)
{
	return helper_get_U32_value(&challenge_verify_cfg);
}

//----------------------------------------------------------------------------------------

//--------------------------------------- internal members -------------------------------
static error_code_t configuration_parse_args(int argc, char *argv[])
{
	error_code_t result=RESULT_OK;
	struct argp argp =
	{
			configuration_options,
			configuration_parse_opt,
			NULL,
			configuration_doc,
			NULL,
			configuration_filter_help,
			NULL
	};

	if (argp_parse(&argp, argc, argv, ARGP_NO_EXIT, 0, &result)==EINVAL)
		result=RESULT_INVALID_ARGS;

	return result;
}

static char *configuration_filter_help(int key, const char *text, void *input)
{
	error_code_t *result_ptr;
	result_ptr=(error_code_t *)input;

	if ((key & ARGP_KEY_HELP_POST_DOC)!=0)
		*result_ptr=RESULT_HELP_PRINTED;
	return (char *)text;
}

static error_t configuration_parse_opt (int key, char *arg, struct argp_state *state)
{
	error_code_t *result_ptr;
	result_ptr=(error_code_t *)state->input;

	switch (key)
	{
	case 'l':
		(*result_ptr) = helper_item_set(&loglevel_cfg, arg);
		if (*result_ptr == RESULT_INVALID_ARGS)
		{
			fprintf(stderr,"Invalid log level in command line.\n");
			argp_usage(state);
		}
		break;
	case 'q':
		console_enabled = false;
		break;
	case 'v':
		configuration_print_version();
		(*result_ptr)=RESULT_HELP_PRINTED;
		break;
	case 'c':
		(*result_ptr) = helper_item_set(&conffile_cfg, arg);
		break;
	default:
		return ARGP_ERR_UNKNOWN;
	}
	return 0;
}


static void configuration_print_version(void)
{
	printf("%s\n\n",VERSION_INFO_FORMATED_LINE);
}
//----------------------------------------------------------------------------------------
